package immibis.core;

import net.minecraft.src.forge.ITextureProvider;
import net.minecraft.src.forge.MinecraftForge;

import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.List;
import java.util.Random;

import net.minecraft.src.BlockContainer;
import net.minecraft.src.EntityItem;
import net.minecraft.src.EntityLiving;
import net.minecraft.src.EntityPlayer;
import net.minecraft.src.IBlockAccess;
import net.minecraft.src.ItemStack;
import net.minecraft.src.Material;
import net.minecraft.src.MathHelper;
import net.minecraft.src.ModLoader;
import net.minecraft.src.NBTTagCompound;
import net.minecraft.src.TileEntity;
import net.minecraft.src.Vec3D;
import net.minecraft.src.World;

public abstract class BlockCombined extends BlockContainer implements ITextureProvider {
	
	private static Random random = new Random();
	
	public ItemCombined item;
	public final String texfile;
	
	public boolean getEnableStats() {return false;}
	
	public BlockCombined(int i, Material m, String texfile) {
		super(i, m);
		this.texfile = texfile;
		if(texfile == null)
			throw new NullPointerException("texfile");
		/* $if client$ */
		net.minecraft.src.forge.MinecraftForgeClient.preloadTexture(texfile);
		try {
			isDefaultTexture = false;
			setTextureFile(texfile);
		} catch(Error e) {
			// Older Forge; that's ok
		}
		/* $endif$ */
		setHardness(2.0F);
	}
	
	public void onBlockPlacedBy(World w, int x, int y, int z, EntityLiving player)
    {
		TileCombined te = (TileCombined)w.getBlockTileEntity(x, y, z);
        
        Vec3D look = player.getLook(1.0f);
        
        double absx = Math.abs(look.xCoord);
        double absy = Math.abs(look.yCoord);
        double absz = Math.abs(look.zCoord);
        
        if(absx > absy && absx > absz) {
        	if(look.xCoord < 0)
        		te.onPlaced(player, Side.NX);
        	else
        		te.onPlaced(player, Side.PX);
        } else if(absy > absz) {
        	if(look.yCoord < 0)
        		te.onPlaced(player, Side.NY);
        	else
        		te.onPlaced(player, Side.PY);
        } else {
        	if(look.zCoord < 0)
        		te.onPlaced(player, Side.NZ);
        	else
        		te.onPlaced(player, Side.PZ);
        }
    }
	
	@Override
	public boolean canProvidePower() {
        return true;
    }
	
	@Override
	public boolean isPoweringTo(IBlockAccess world, int x, int y, int z, int i) {
		TileCombined te = (TileCombined)world.getBlockTileEntity(x, y, z);
		if(te != null)
			return te.redstone_output;
		return false;
	}
	
	@Override
	public boolean isIndirectlyPoweringTo(World world, int x, int y, int z, int i) {
		return isPoweringTo(world, x, y, z, i);
	}
	
	@Override
	public ArrayList getBlockDropped(World world, int i, int j, int k, int l, int i1)
	{
		ArrayList list = new ArrayList();
		list.add(new ItemStack(this, 1, l));
		TileEntity te = world.getBlockTileEntity(i, j, k);
		//System.out.println("getBlockDropped(" + te + ")");
		if(te != null && te instanceof TileBasicInventory)
		{
			BasicInventory inv = ((TileBasicInventory)te).inv;
			for(ItemStack stack : inv.contents)
				if(stack != null && stack.stackSize > 0)
					list.add(stack);
		}
		return list;
	}
	
	@Override
	public abstract int getBlockTextureFromSideAndMetadata(int side, int data);
	
	@Override
	public int getRenderType() {
		return 0;
	}
	
	@Override
	public void onNeighborBlockChange(World w, int x, int y, int z, int i)
	{
		if(w.isRemote)
			return;
		TileCombined te = (TileCombined)w.getBlockTileEntity(x, y, z);
		if(te != null)
			te.onBlockNeighbourChange();
	}
	
	@Override
	public boolean blockActivated(World world, int x, int y, int z, EntityPlayer player)
    {
		if(player.isSneaking())
			return false;
		TileCombined te = (TileCombined)world.getBlockTileEntity(x, y, z);
		if(te != null)
			return te.onBlockActivated(player);
		return false;
    }
	
	@Override
	public void onBlockRemoval(World world, int x, int y, int z)
    {
		TileCombined te = (TileCombined)world.getBlockTileEntity(x, y, z);
		if(te != null)
			te.onBlockRemoval();
		
		List<ItemStack> drops = getBlockDropped(world, x, y, z, -1, -1);
		drops.remove(0); // remove the block itself
		for(ItemStack stack : drops) {
			float xpos = x + random.nextFloat() * 0.8f + 0.1f;
			float ypos = y + random.nextFloat() * 0.8f + 0.1f;
			float zpos = z + random.nextFloat() * 0.8f + 0.1f;
			
			//System.out.println("drop "+stack + " at "+xpos+","+ypos+","+zpos);
			
			// chests do this (multiple drops per stack, 10-30 items at a time)
			int left = stack.stackSize;
			while(left > 0) {
				int removeCount = Math.min(random.nextInt(21) + 10, left);
				left -= removeCount;
				
				EntityItem ent = new EntityItem(world, xpos, ypos, zpos, new ItemStack(stack.itemID, removeCount, stack.getItemDamage()));
				
				ent.motionX = random.nextGaussian() * 0.05f;
				ent.motionY = random.nextGaussian() * 0.05f + 0.2f;
				ent.motionZ = random.nextGaussian() * 0.05f;
				
				if(stack.hasTagCompound())
					ent.item.setTagCompound(stack.getTagCompound());
				
				world.spawnEntityInWorld(ent);
	        }
		}
		super.onBlockRemoval(world, x, y, z);
    }
	
	@Override
	public abstract TileEntity getBlockEntity(int data);

	@Override
	public TileEntity getBlockEntity() {
		throw new UnsupportedOperationException("");
	}

	@Override
	public String getTextureFile() {
		if(texfile == null)
			return "/terrain.png"; // work around a bug in newer Forge's Block.<init>
		return texfile;
	}
	
	@Override
    public abstract void addCreativeItems(ArrayList arraylist);
}
